/* 
 *  InteractionData.java
 * 
 *  Copyright 2016 Avaya Inc. All Rights Reserved.
 * 
 *  Usage of this source is bound to the terms described
 *  in AvayaLicenseSDK.rtf.
 * 
 *  Avaya - Confidential & Proprietary. Use pursuant to your signed agreement
 *  or Avaya Policy
 * 
 */
package com.avaya.ccs.javafxrefclient;

import com.avaya.ccs.api.ContactCenterCodeI;
import com.avaya.ccs.api.ContactNumberI;
import com.avaya.ccs.api.CustomerDetailsI;
import com.avaya.ccs.api.DataI;
import com.avaya.ccs.api.InteractionI;
import com.avaya.ccs.api.IntrinsicI;
import com.avaya.ccs.api.RemoteParticipantI;
import com.avaya.ccs.api.enums.ContactType;
import com.avaya.ccs.api.enums.DestinationType;
import com.avaya.ccs.api.enums.InteractionState;
import com.avaya.ccs.api.enums.StateReason;
import com.avaya.ccs.api.exceptions.InvalidArgumentException;
import com.avaya.ccs.api.exceptions.ObjectInvalidException;
import com.avaya.ccs.core.BaseInteraction;
import com.avaya.ccs.core.CodeList;
import com.avaya.ccs.core.CustomFieldList;
import com.avaya.ccs.core.EmailMedia;
import com.avaya.ccs.core.ScreenPop;
import com.fasterxml.jackson.annotation.JsonProperty;

import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import javafx.util.Pair;

public class InteractionData extends Data implements InteractionDataI {
	
    // Capabilities
    private boolean canAnswer = false;
    private boolean canCompleteConference = false;
    private boolean canCompleteTransfer = false;
    private boolean canEmergency = false;
    private boolean canEnd = false;
    private boolean canHold = false;
    private boolean canInitiateConference = false;
    private boolean canInitiateTransfer = false;
    private boolean canJoinConference = false;
    private boolean canPlayDTMF = false;
    private boolean canSetActivityCode = false;
    private boolean canSetUUI = false;
    private boolean canUnhold = false;
    private boolean canTransferToDestination = false;
    // POM Capbilities
    private boolean canAddAgentNote = false;
    private boolean canRefreshAgentNotes = false;
    private boolean canGetCallbackDestinations = false;
    private boolean canGetConsultDestinations = false;
    private boolean canGetDispositionCodes = false;
    private boolean canWrapupCall = false;
    private boolean canWrapupCallWithCode = false;
    private boolean canExtendWrapup = false;
    private boolean canCreateCallback = false;
    private boolean canPreviewDialFreeForm = false;
    private boolean canPreviewDial = false;
    private boolean canPreviewCancel = false;
    private boolean canRedial = false;
    private boolean canInitiateTransferToDestination = false;
    private boolean canInitiateConferenceToDestination = false;
    private boolean canChangeConferenceOwner = false;
    private boolean canEndConference = false;
    private boolean canEndConsult = false;
    private boolean canUpdateCustomerName = false;
    private boolean canUpdateCustomerAddress = false;
    private boolean canUpdateCustomerPhone = false;
    private boolean canUpdateCustomerEmail = false;
    private boolean canAddToDnc = false;
    private boolean canOpenMedia =false;
    private boolean canGetAutoPhrases = false;
    private boolean canGetPagePushUrls = false;
    private boolean canGetCustomerHistory =false;
    private boolean canGetActivityCodes = false;

    // Properties
    private ContactCenterCodeI activityCode = null;
    private String activityCodeNumber = null;
    private String address = "";
    private String addressName = "";
    private String calledAddress = "";
    private String calledAddressName = "";
    private String callingAddress = "";
    private String callingAddressName = "";
    private String contactID = "";
    private ContactType contactType = ContactType.Unknown;
    private DataI data = null;
    private String id = "";
    private List<IntrinsicElement> intrinsicsElements = new LinkedList<>();
    private String mainContactID = "";
    private String originalDestination = "";
    private String resourceID = "";
    private String skillset = "";
    private InteractionState state = InteractionState.Unknown;
    private StateReason stateReason = StateReason.Unknown;
    private String uui = "";
    private String userID = "";
    private boolean monitored = false;
    private boolean multimedia = false;
    
    private boolean isConsult;
    private boolean isConference;
    
    
    private StateReason participationReason = StateReason.Unknown;
    private List<RemoteParticipantI> remoteParticipants = new LinkedList<>();

    // POM properties
    private CustomerDetailsI customerDetails = null;
    private List<String> agentNotes = new LinkedList<>();
    private int wrapupTimer = 0;
    private int previewTimer = 0;
    private String campaignName = "";
    private String agentScriptUrl = "";
    private List<ContactNumberI> contactNumbers = new LinkedList<>();
    private List<DestinationType> consultDestinationTypes = new LinkedList<>();
    private List<DestinationType> callbackDestinationTypes = new LinkedList<>();
    private List<ScreenPop> screenPops = new ArrayList<ScreenPop>();
    CodeList dispositionCodes;
    CustomFieldList customFields;

    // timeLeft field is used to hold the time remaining on a preview of wrapup
    // timer
    private int timeLeft = 0;

    public InteractionData(InteractionI interaction) {
    	
        if (interaction != null) {
            try {
                // Capabilities
                canAnswer = interaction.canAnswer();
                canEmergency = interaction.canEmergency();
                canEnd = interaction.canEnd();
                canHold = interaction.canHold();
                canUnhold = interaction.canUnhold();
                canInitiateConference = interaction.canInitiateConference();
                canCompleteConference = interaction.canCompleteConference();
                canInitiateTransfer = interaction.canInitiateTransfer();
                canCompleteTransfer = interaction.canCompleteTransfer();
                canJoinConference = interaction.canJoinConference();
                canPlayDTMF = interaction.canPlayDTMF();
                canSetActivityCode = interaction.canSetActivityCode();
                canSetUUI = interaction.canSetUUI();
                // achokoev - 09/17/2019 - CCS Java API does not provide a canTransferToDestination capability on interactions
                canTransferToDestination = interaction.canTransferToDestination();

                // POM capabilities
                canAddAgentNote = interaction.canAddAgentNote();
                canRefreshAgentNotes = interaction.canRefreshAgentNotes();
                canGetCallbackDestinations = interaction.canGetCallbackDestinations();
                canGetConsultDestinations = interaction.canGetConsultDestinations();
                canGetDispositionCodes = interaction.canGetDispositionCodes();
                canWrapupCall = interaction.canWrapup();
                canWrapupCallWithCode = interaction.canWrapupWithCode();
                canExtendWrapup = interaction.canExtendWrapup();
                canCreateCallback = interaction.canCreateCallback();
                canPreviewDialFreeForm = interaction.canPreviewDialFreeForm();
                canPreviewDial = interaction.canPreviewDial();
                canPreviewCancel = interaction.canPreviewCancel();
                canRedial = interaction.canRedial();
                canInitiateTransferToDestination = interaction.canInitiateTransferToDestination();
                canInitiateConferenceToDestination = interaction.canInitiateConferenceToDestination();
                canChangeConferenceOwner = interaction.canChangeConferenceOwner();
                canEndConference = interaction.canEndConference();
                canEndConsult=interaction.canEndConsult();
                canUpdateCustomerName = interaction.canUpdateCustomerName();
                canUpdateCustomerAddress = interaction.canUpdateCustomerAddress();
                canUpdateCustomerPhone = interaction.canUpdateCustomerPhone();
                canUpdateCustomerEmail = interaction.canUpdateCustomerEmail();
                canAddToDnc = interaction.canAddToDnc();
                canOpenMedia = interaction.canOpenMedia();
                canGetAutoPhrases = interaction.canGetAutoPhrases();
                canGetPagePushUrls = interaction.canGetPagePushUrls();
                canGetCustomerHistory = interaction.canGetCustomerHistory();
                canGetActivityCodes = interaction.canGetActivityCodes();

                // Properties
                activityCode = interaction.getActivityCode();
                if (activityCode != null) {
                    activityCodeNumber = activityCode.getNumber().toString();
                }
                address = interaction.getAddress();
                addressName = interaction.getAddressName();
                calledAddress = interaction.getCalledAddress();
                calledAddressName = interaction.getCalledAddressName();
                callingAddress = interaction.getCallingAddress();
                callingAddressName = interaction.getCallingAddressName();
                contactID = interaction.getContactId();
                contactType = interaction.getContactType();
                data = interaction.getData();
                id = interaction.getId();
                intrinsicsElements = buildIntrinsicsElmentList(interaction);
                mainContactID = interaction.getMainContactId();
                originalDestination = interaction.getOriginalDestination();
                resourceID = interaction.getResourceId();
                skillset = interaction.getSkillset();
                state = interaction.getState();
                stateReason = interaction.getStateReason();
                uui = interaction.getUUI();
                userID = interaction.getUserId();
                monitored = interaction.isMonitored();
                multimedia = interaction.isMultimedia();
                isConsult = interaction.isConsult();
                isConference = interaction.isConference();
                participationReason = interaction.getParticipationReason();
                remoteParticipants = interaction.getRemoteParticipants();
                screenPops = interaction.getScreenPops();
                
                
                // POM properties
                customerDetails = interaction.getCustomerDetails();
                agentNotes = interaction.getAgentNotes();
                wrapupTimer = interaction.getWrapupTimer();
                previewTimer = interaction.getPreviewTimer();
                campaignName = interaction.getCampaignName();
                agentScriptUrl = interaction.getAgentScriptUrl();
                contactNumbers = interaction.getContactNumbers();
                callbackDestinationTypes = interaction.getCallbackDestinationTypes();
                
            } catch (ObjectInvalidException ex) {
                initializeToDefault();
            }
        }
    }

    private void initializeToDefault() {
        canAnswer = false;
        canCompleteConference = false;
        canCompleteTransfer = false;
        canEmergency = false;
        canEnd = false;
        canHold = false;
        canInitiateConference = false;
        canInitiateTransfer = false;
        canJoinConference = false;
        canPlayDTMF = false;
        canSetActivityCode = false;
        canSetUUI = false;
        canUnhold = false;
        canAddAgentNote = false;
        canRefreshAgentNotes = false;
        canGetCallbackDestinations = false;
        canGetConsultDestinations = false;
        canGetDispositionCodes = false;
        canWrapupCall = false;
        canWrapupCallWithCode = false;
        canExtendWrapup = false;
        canCreateCallback = false;
        canPreviewDialFreeForm = false;
        canPreviewDial = false;
        canPreviewCancel = false;
        canRedial = false;
        canInitiateTransferToDestination = false;
        canInitiateConferenceToDestination = false;
        canChangeConferenceOwner = false;
        canEndConference = false;
        canUpdateCustomerName = false;
        canUpdateCustomerAddress = false;
        canUpdateCustomerPhone = false;
        canUpdateCustomerEmail = false;
        canAddToDnc = false;

        // Properties
        activityCode = null;
        activityCodeNumber = null;
        address = "";
        addressName = "";
        calledAddress = "";
        calledAddressName = "";
        callingAddress = "";
        callingAddressName = "";
        contactID = "";
        contactType = ContactType.Unknown;
        data = null;
        id = "";
        intrinsicsElements = new LinkedList<>();
        mainContactID = "";
        originalDestination = "";
        resourceID = "";
        skillset = "";
        state = InteractionState.Unknown;
        stateReason = StateReason.Unknown;
        uui = "";
        userID = "";
        monitored = false;
        multimedia = false;
        isConference=false;
        isConsult=false;
        participationReason = StateReason.Unknown;
        remoteParticipants = new LinkedList<>();
        customerDetails = null;
        agentNotes = new LinkedList<>();
        contactNumbers = new LinkedList<>();
        consultDestinationTypes = new LinkedList<>();
        callbackDestinationTypes = new LinkedList<>();
        wrapupTimer = 0;
        previewTimer = 0;
    }

    public InteractionData(String id, boolean deleted) {
        this.id = id;
        this.setDeleted(deleted);
    }

    @Override
    public List<Pair<String, String>> getProperties() {
        // Properties
        List<Pair<String, String>> properties = new LinkedList<>();
        properties.add(new Pair<String, String>("ID", this.getId()));
        String actCode = "NONE";
        if (this.activityCode != null) {
            actCode = this.activityCode.getName() + ":" + this.activityCode.getNumber();
        }

        properties.add(new Pair<String, String>("ActivityCode", actCode));
        properties.add(new Pair<String, String>("Address", this.getAddress()));
        properties.add(new Pair<String, String>("AddressName", this.getAddressName()));
        properties.add(new Pair<String, String>("CalledAddress", this.getCalledAddress()));
        properties.add(new Pair<String, String>("CalledAddressName", this.getCalledAddressName()));
        properties.add(new Pair<String, String>("CallingAddress", this.getCalledAddress()));
        properties.add(new Pair<String, String>("CallingAddressName", this.getCallingAddressName()));
        properties.add(new Pair<String, String>("ContactId", this.getContactID()));
        properties.add(new Pair<String, String>("ContactType", this.getContactType().name()));
        properties.add(new Pair<String, String>("MainContactId", this.getMainContactID()));
        properties.add(new Pair<String, String>("OriginalDestination", this.getOriginalDestination()));
        properties.add(new Pair<String, String>("ResourceId", this.getResourceID()));
        properties.add(new Pair<String, String>("Skillset", this.getSkillset()));
        properties.add(new Pair<String, String>("UserId", this.getUserID()));
        properties.add(new Pair<String, String>("isMonitored", Boolean.toString(this.isMonitored())));
        properties.add(new Pair<String, String>("isMultimedia", Boolean.toString(this.isMultimedia())));
        properties.add(new Pair<String, String>("State", this.getState().name()));
        properties.add(new Pair<String, String>("StateReason", this.getStateReason().name()));
        properties.add(new Pair<String, String>("ParticipationReason", this.getParticipationReasonString()));
        // Add remote participants as a list
        int count = 1;
        for (RemoteParticipantI participant : remoteParticipants) {
            properties.add(new Pair<String, String>("particpant:" + count + ":Address", participant.getAddress()));
            properties.add(new Pair<String, String>("particpant:" + count + ":Reason", participant.getReason().name()));
            properties.add(new Pair<String, String>("particpant:" + count + ":State", participant.getState().name()));
            count++;
        }
        
        properties.add(new Pair<String, String>("IsConsult", Boolean.toString(this.isConsult)));
        properties.add(new Pair<String, String>("IsConference", Boolean.toString(this.isConference)));


        // POM properties
        properties.add(new Pair<String, String>("wrapupTimer", Integer.toString(this.getWrapupTimer())));
        properties.add(new Pair<String, String>("previewTimer", Integer.toString(this.getPreviewTimer())));
        properties.add(new Pair<String, String>("campaignName", this.getCampaignName()));
        properties.add(new Pair<String, String>("agentScriptUrl", this.getAgentScriptUrl()));

        return properties;
    }

    @Override
    public List<Pair<String, Boolean>> getCaps() {
        List<Pair<String, Boolean>> caps = new LinkedList<>();
        caps.add(new Pair<String, Boolean>("canAnswer", this.canAnswer()));
        caps.add(new Pair<String, Boolean>("canCompleteConference", this.canCompleteConference()));
        caps.add(new Pair<String, Boolean>("canCompleteTransfer", this.canCompleteTransfer()));
        caps.add(new Pair<String, Boolean>("canEmergency", this.canEmergency()));
        caps.add(new Pair<String, Boolean>("canEnd", this.canEnd()));
        caps.add(new Pair<String, Boolean>("canHold", this.canHold()));
        caps.add(new Pair<String, Boolean>("canInitiateConference", this.canInitiateConference()));
        caps.add(new Pair<String, Boolean>("canInitiateTransfer", this.canInitiateTransfer()));
        caps.add(new Pair<String, Boolean>("canJoinConference", this.canJoinConference()));
        caps.add(new Pair<String, Boolean>("canPlayDTMF", this.canPlayDTMF()));
        caps.add(new Pair<String, Boolean>("canSetActivityCode", this.canSetActivityCode()));
        caps.add(new Pair<String, Boolean>("canSetUUI", this.canSetUUI()));
        caps.add(new Pair<String, Boolean>("canUnhold", this.canUnhold()));
        caps.add(new Pair<String, Boolean>("canAddAgentNote", this.canAddAgentNote()));
        caps.add(new Pair<String, Boolean>("canRefreshAgentNotes", this.canRefreshAgentNotes()));
        caps.add(new Pair<String, Boolean>("canGetCallbackDestinations", this.canGetCallbackDestinations()));
        caps.add(new Pair<String, Boolean>("canGetConsultDestinations", this.canGetConsultDestinations()));
        caps.add(new Pair<String, Boolean>("canGetDispositionCodes", this.canGetDispositionCodes()));
        caps.add(new Pair<String, Boolean>("canWrapupCall", this.canWrapupCall()));
        caps.add(new Pair<String, Boolean>("canWrapupCallWithCode", this.canWrapupCallWithCode()));
        caps.add(new Pair<String, Boolean>("canExtendWrapup", this.canExtendWrapup()));
        caps.add(new Pair<String, Boolean>("canCreateCallback", this.canCreateCallback()));
        caps.add(new Pair<String, Boolean>("canEnterFreeFormNumber", this.canPreviewDialFreeForm()));
        caps.add(new Pair<String, Boolean>("canPreviewDial", this.canPreviewDial()));
        caps.add(new Pair<String, Boolean>("canPreviewCancel", this.canPreviewCancel()));
        caps.add(new Pair<String, Boolean>("canRedial", this.canRedial()));
        caps.add(
                new Pair<String, Boolean>("canInitiateTransferToDestination", this.canInitiateTransferToDestination()));
        caps.add(new Pair<String, Boolean>("canInitiateConferenceToDestination",
                this.canInitiateConferenceToDestination()));
        caps.add(new Pair<String, Boolean>("canChangeConferenceOwner", this.canChangeConferenceOwner()));
        caps.add(new Pair<String, Boolean>("canEndConference", this.canEndConference()));
        caps.add(new Pair<String, Boolean>("canUpdateCustomerName", this.canUpdateCustomerName()));
        caps.add(new Pair<String, Boolean>("canUpdateCustomerAddress", this.canUpdateCustomerAddress()));
        caps.add(new Pair<String, Boolean>("canUpdateCustomerPhone", this.canUpdateCustomerPhone()));
        caps.add(new Pair<String, Boolean>("canUpdateCustomerEmail", this.canUpdateCustomerEmail()));
        caps.add(new Pair<String, Boolean>("canAddToDnc", this.canAddToDnc()));

        return caps;
    }

    private List<IntrinsicElement> buildIntrinsicsElmentList(InteractionI interaction) {
        List<IntrinsicElement> l = new ArrayList<>();        
        try {
            for (IntrinsicI i : interaction.getIntrinsics()) {
            	l.add(new IntrinsicElement(i.getName(), i.getValue()));
            }
        } catch (ObjectInvalidException e) {
            return new ArrayList<>();
        } 
        return l;
    }

    @Override
    public String getId() {
        return this.id;
    }

    public boolean canAnswer() {
        return canAnswer;
    }

    public boolean canCompleteConference() {
        return canCompleteConference;
    }

    public boolean canCompleteTransfer() {
        return canCompleteTransfer;
    }

    public boolean canEmergency() {
        return canEmergency;
    }

    public boolean canEnd() {
        return canEnd;
    }

    public boolean canHold() {
        return canHold;
    }

    public boolean canInitiateConference() {
        return canInitiateConference;
    }

    public boolean canInitiateTransfer() {
        return canInitiateTransfer;
    }

    public boolean canJoinConference() {
        return canJoinConference;
    }

    public boolean canPlayDTMF() {
        return canPlayDTMF;
    }

    public boolean canSetActivityCode() {
        return canSetActivityCode;
    }

    public boolean canSetUUI() {
        return canSetUUI;
    }

    public boolean canUnhold() {
        return canUnhold;
    }

    public boolean canSetIntrinsic() {
        return contactType.equals(ContactType.Voice);
    }

    public boolean canGetIntrinsic() {
        return contactType.equals(ContactType.Voice);
    }

    public boolean canGetUUI() {
        return contactType.equals(ContactType.Voice);
    }

    public boolean canGetData() {
        return contactType.equals(ContactType.Voice);
    }

    public boolean canGetCustomerDetails() {
    	return true;
    }

    public ContactCenterCodeI getActivityCode() {
        return activityCode;
    }

    public String getAddress() {
        return address;
    }

    public String getAddressName() {
        return addressName;
    }

    public String getCalledAddress() {
        return calledAddress;
    }

    public String getCalledAddressName() {
        return calledAddressName;
    }

    public String getCallingAddress() {
        return callingAddress;
    }

    public String getCallingAddressName() {
        return callingAddressName;
    }

    public String getContactID() {
        return contactID;
    }

    public ContactType getContactType() {
        return contactType;
    }

    public DataI getData() {
        return data;
    }

    public List<IntrinsicElement> getIntrinsics() {
        return intrinsicsElements;
    }

    public String getMainContactID() {
        return mainContactID;
    }

    public String getOriginalDestination() {
        return originalDestination;
    }

    public String getResourceID() {
        return resourceID;
    }

    public String getSkillset() {
        return skillset;
    }

    public InteractionState getState() {
        return state;
    }

    public StateReason getStateReason() {
        return stateReason;
    }

    public String getCombinedState() {
        String combinedValue = state.toString();
        switch (state) {
        case Preview:
        case Wrapup:
            if (timeLeft > 0) {
                combinedValue += "(" + Integer.toString(timeLeft) + ")";
            }
            break;
        default:
            break;
        }

        return combinedValue;
    }

    public String getUui() {
        return uui;
    }

    public String getUserID() {
        return userID;
    }

    public String getActivityCodeNumber() {
        return activityCodeNumber;
    }

    public boolean canAddAgentNote() {
        return canAddAgentNote;
    }

    public boolean canRefreshAgentNotes() {
        return canRefreshAgentNotes;
    }

    public boolean canGetCallbackDestinations() {
        return canGetCallbackDestinations;
    }

    public boolean canGetConsultDestinations() {
        return canGetConsultDestinations;
    }

    public boolean canGetDispositionCodes() {
        return canGetDispositionCodes;
    }
    
    public boolean canGetActivityCodes() {
        return canGetActivityCodes;
    }

    public boolean canWrapupCall() {
        return canWrapupCall;
    }

    public boolean canWrapupCallWithCode() {
        return canWrapupCallWithCode;
    }

    public boolean canExtendWrapup() {
        return canExtendWrapup;
    }

    public boolean canCreateCallback() {
        return canCreateCallback;
    }

    public boolean canPreviewDialFreeForm() {
        return canPreviewDialFreeForm;
    }

    public boolean canPreviewDial() {
        return canPreviewDial;
    }

    public boolean canPreviewCancel() {
        return canPreviewCancel;
    }

    public boolean canRedial() {
        return canRedial;
    }

    public boolean canInitiateTransferToDestination() {
        return canInitiateTransferToDestination;
    }

    public boolean canInitiateConferenceToDestination() {
        return canInitiateConferenceToDestination;
    }

    public boolean canChangeConferenceOwner() {
        return canChangeConferenceOwner;
    }

    public boolean canEndConference() {
        return canEndConference;
    }
    
    public boolean canEndConsult() {
        return canEndConsult;
    }

    public boolean canUpdateCustomerName() {
        return canUpdateCustomerName;
    }

    public boolean canUpdateCustomerAddress() {
        return canUpdateCustomerAddress;
    }

    public boolean canUpdateCustomerPhone() {
        return canUpdateCustomerPhone;
    }

    public boolean canUpdateCustomerEmail() {
        return canUpdateCustomerEmail;
    }

    public boolean canAddToDnc() {
        return canAddToDnc;
    }
    
	public boolean canOpenMedia() {
		return canOpenMedia;
	}
	
	public boolean canGetAutoPhrases() {
		return canGetAutoPhrases;
	}
	
	public boolean canGetPagePushUrls() {
		return canGetPagePushUrls;
	}
	
	public boolean canGetCustomerHistory() {
		return canGetCustomerHistory;
	}
	

    public boolean isMonitored() {
        return monitored;
    }

    public boolean isMultimedia() {
        return multimedia;
    }
    
    public boolean isConference() {
    	return isConference;
    }
   
    public boolean isConsult() {
    	return isConsult;
    }

    public StateReason getParticipationReason() {
        return participationReason;
    }

    public List<RemoteParticipantI> getRemoteParticipants() {
        return remoteParticipants;
    }

    public CustomerDetailsI getCustomerDetails() {
        return customerDetails;
    }

    public List<ScreenPop> getScreenPops() {
    	return screenPops;
    }
    public List<String> getAgentNotes() {
        return agentNotes;
    }

    public int getPreviewTimer() {
        return previewTimer;
    }

    public int getWrapupTimer() {
        return wrapupTimer;
    }

    public String getCampaignName() {
        return campaignName;
    }

    public String getAgentScriptUrl() {
        return agentScriptUrl;
    }

    public List<ContactNumberI> getContactNumbers() {
        return contactNumbers;
    }

    public ContactNumberI getDefaultOrFirstContactNumber() {
        for (ContactNumberI number : getContactNumbers()) {
            if (number.isDefault())
                return number;
        }
        return getContactNumbers().get(0);
    }

    public List<DestinationType> getConsultDestinationTypes() {
        return consultDestinationTypes;
    }

    public List<DestinationType> getCallbackDestinationTypes() {
        return callbackDestinationTypes;
    }

    public String getParticipationReasonString() {
        // participation reason will be null in most cases
        if (this.getParticipationReason() == null) {
            return StateReason.Unknown.name();
        }
        return this.getParticipationReason().name();
    }

    public void setTimeLeft(int timeLeft) {
        this.timeLeft = timeLeft;
    }

    public int getTimeLeft() {
        return timeLeft;
    }
    
    
    public CodeList getDispositionCodes() {
 		return dispositionCodes;
 	}

 	public void setDispositionCodes(CodeList dispositionCodes) {
 		this.dispositionCodes = dispositionCodes;
 	}
    
    public CustomFieldList getCustomFields() {
        return customFields;
    }

    public void setCustomFields(CustomFieldList customFields) {
        this.customFields = customFields;
    }

    @Override
    /**
     * This methods is used to update the object with new data, this can be
     * useful for updating an observable list
     */
    public void update(Data newData) {
        InteractionData newInteractionData = (InteractionData) newData;
        
        // Capabilities
        this.canAnswer = newInteractionData.canAnswer();
        this.canCompleteConference = newInteractionData.canCompleteConference();
        this.canCompleteTransfer = newInteractionData.canCompleteTransfer();
        this.canEmergency = newInteractionData.canEmergency();
        this.canEnd = newInteractionData.canEnd();
        this.canHold = newInteractionData.canHold();
        this.canInitiateConference = newInteractionData.canInitiateConference();
        this.canInitiateTransfer = newInteractionData.canInitiateTransfer();
        this.canJoinConference = newInteractionData.canJoinConference();
        this.canPlayDTMF = newInteractionData.canPlayDTMF();
        this.canSetActivityCode = newInteractionData.canSetActivityCode();
        this.canSetUUI = newInteractionData.canSetUUI();
        this.canUnhold = newInteractionData.canUnhold();
        // POM Capbilities
        this.canAddAgentNote = newInteractionData.canAddAgentNote();
        this.canRefreshAgentNotes = newInteractionData.canRefreshAgentNotes();
        this.canGetCallbackDestinations = newInteractionData.canGetCallbackDestinations();
        this.canGetConsultDestinations = newInteractionData.canGetConsultDestinations();
        this.canGetDispositionCodes = newInteractionData.canGetDispositionCodes();
        this.canWrapupCall = newInteractionData.canWrapupCall();
        this.canWrapupCallWithCode = newInteractionData.canWrapupCallWithCode();
        this.canExtendWrapup = newInteractionData.canExtendWrapup();
        this.canCreateCallback = newInteractionData.canCreateCallback();
        this.canPreviewDialFreeForm = newInteractionData.canPreviewDialFreeForm();
        this.canPreviewDial = newInteractionData.canPreviewDial();
        this.canPreviewCancel = newInteractionData.canPreviewCancel();
        this.canRedial = newInteractionData.canRedial();
        this.canInitiateTransferToDestination = newInteractionData.canInitiateTransferToDestination();
        this.canInitiateConferenceToDestination = newInteractionData.canInitiateConferenceToDestination();
        this.canChangeConferenceOwner = newInteractionData.canChangeConferenceOwner();
        this.canEndConference = newInteractionData.canEndConference();
        this.canUpdateCustomerName = newInteractionData.canUpdateCustomerName();
        this.canUpdateCustomerAddress = newInteractionData.canUpdateCustomerAddress();
        this.canUpdateCustomerPhone = newInteractionData.canUpdateCustomerPhone();
        this.canUpdateCustomerEmail = newInteractionData.canUpdateCustomerEmail();
        this.canAddToDnc = newInteractionData.canAddToDnc();

        // Properties
        this.activityCode = newInteractionData.getActivityCode();
        this.activityCodeNumber = newInteractionData.getActivityCodeNumber();
        this.address = newInteractionData.getAddress();
        this.addressName = newInteractionData.getAddressName();
        this.calledAddress = newInteractionData.getCalledAddress();
        this.calledAddressName = newInteractionData.getCalledAddressName();
        this.callingAddress = newInteractionData.getCallingAddress();
        this.callingAddressName = newInteractionData.getCallingAddressName();
        this.contactID = newInteractionData.getContactID();
        this.contactType = newInteractionData.getContactType();
        this.data = newInteractionData.getData();
        this.id = newInteractionData.getId();
        this.intrinsicsElements = newInteractionData.getIntrinsics();
        this.mainContactID = newInteractionData.getMainContactID();
        this.originalDestination = newInteractionData.getOriginalDestination();
        this.resourceID = newInteractionData.getResourceID();
        this.skillset = newInteractionData.getSkillset();
        this.state = newInteractionData.getState();
        this.stateReason = newInteractionData.getStateReason();
        this.uui = newInteractionData.getUui();
        this.userID = newInteractionData.getUserID();
        this.monitored = newInteractionData.isMonitored();
        this.multimedia = newInteractionData.isMultimedia();
        this.participationReason = newInteractionData.getParticipationReason();
        this.remoteParticipants = newInteractionData.getRemoteParticipants();
        this.isConsult = newInteractionData.isConsult();
        this.isConference=newInteractionData.isConference();

        // POM properties
        this.customerDetails = newInteractionData.getCustomerDetails();
        this.agentNotes = newInteractionData.getAgentNotes();
        this.wrapupTimer = newInteractionData.getWrapupTimer();
        this.previewTimer = newInteractionData.getPreviewTimer();
        this.campaignName = newInteractionData.getCampaignName();
        this.agentScriptUrl = newInteractionData.getAgentScriptUrl();
        this.contactNumbers = newInteractionData.getContactNumbers();
        this.consultDestinationTypes = newInteractionData.getConsultDestinationTypes();
        this.callbackDestinationTypes = newInteractionData.getCallbackDestinationTypes();
        this.timeLeft = newInteractionData.getTimeLeft();
        
        
        newInteractionData.getDispositionCodes();
    }
}
